/*
 * Source file of the Halachic Times project.
 * Copyright (c) 2012. All Rights Reserved.
 * 
 * The contents of this file are subject to the Mozilla Public License Version
 * 2.0 (the "License"); you may not use this file except in compliance with the
 * License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/2.0
 *
 * Contributors can be contacted by electronic mail via the project Web pages:
 * 
 * http://sourceforge.net/projects/halachictimes
 * 
 * http://halachictimes.sourceforge.net
 *
 * Contributor(s):
 *   Moshe Waisberg
 * 
 */
package net.sf.times;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.Locale;

import net.sf.times.ZmanimAdapter.ZmanimItem;
import net.sf.times.location.ZmanimLocations;
import net.sourceforge.zmanim.ComplexZmanimCalendar;
import net.sourceforge.zmanim.hebrewcalendar.HebrewDateFormatter;
import net.sourceforge.zmanim.hebrewcalendar.JewishCalendar;
import net.sourceforge.zmanim.hebrewcalendar.JewishDate;
import net.sourceforge.zmanim.util.GeoLocation;
import android.content.Context;
import android.content.res.Resources;
import android.text.format.DateUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ArrayAdapter;
import android.widget.TextView;

/**
 * Adapter for halachic times list.
 * <p>
 * See also Wikipedia article on <a
 * href="http://en.wikipedia.org/wiki/Zmanim">Zmanim</a>.
 * 
 * @author Moshe Waisberg
 */
public class ZmanimAdapter extends ArrayAdapter<ZmanimItem> {

	/** 12 hours (half a full day). */
	protected static final long TWELVE_HOURS = DateUtils.DAY_IN_MILLIS >> 1;

	/** Holiday id for Shabbath. */
	public static final int SHABBATH = 100;

	/** No candles to light. */
	private static final int CANDLES_NONE = 0;
	/** Number of candles to light for Shabbath. */
	private static final int CANDLES_SHABBATH = 2;
	/** Number of candles to light for a festival. */
	private static final int CANDLES_FESTIVAL = 2;
	/** Number of candles to light for Yom Kippurim. */
	private static final int CANDLES_YOM_KIPPUR = 1;

	/** Flag indicating lighting times before sunset. */
	private static final int BEFORE_SUNSET = 0x00000000;
	/** Flag indicating lighting times at sunset. */
	private static final int AT_SUNSET = 0x10000000;
	/** Flag indicating lighting times after nightfall. */
	private static final int MOTZE_SHABBATH = 0x20000000;

	protected static final int CANDLES_MASK = 0x0000000F;
	protected static final int HOLIDAY_MASK = 0x000000FF;
	protected static final int MOTZE_MASK = 0xF0000000;

	private static final String OPINION_10_2 = "10.2";
	private static final String OPINION_11 = "11";
	private static final String OPINION_12 = "12";
	private static final String OPINION_120 = "120";
	private static final String OPINION_120_ZMANIS = "120_zmanis";
	private static final Object OPINION_13 = "13.24";
	private static final String OPINION_15 = "15";
	private static final String OPINION_16_1 = "16.1";
	private static final String OPINION_16_1_ALOS = "16.1_alos";
	private static final String OPINION_16_1_SUNSET = "16.1_sunset";
	private static final String OPINION_18 = "18";
	private static final String OPINION_19_8 = "19.8";
	private static final String OPINION_2 = "2";
	private static final String OPINION_26 = "26";
	private static final String OPINION_3 = "3";
	private static final String OPINION_3_65 = "3.65";
	private static final String OPINION_3_676 = "3.676";
	private static final String OPINION_30 = "30";
	private static final String OPINION_4_37 = "4.37";
	private static final String OPINION_4_61 = "4.61";
	private static final String OPINION_4_8 = "4.8";
	private static final String OPINION_5_88 = "5.88";
	private static final String OPINION_5_95 = "5.95";
	private static final Object OPINION_58 = "58.5";
	private static final String OPINION_60 = "60";
	private static final String OPINION_7 = "7";
	private static final String OPINION_7_083 = "7.083";
	private static final String OPINION_72 = "72";
	private static final String OPINION_72_ZMANIS = "72_zmanis";
	private static final String OPINION_8_5 = "8.5";
	private static final String OPINION_90 = "90";
	private static final String OPINION_90_ZMANIS = "90_zmanis";
	private static final String OPINION_96 = "96";
	private static final String OPINION_96_ZMANIS = "96_zmanis";
	private static final String OPINION_ATERET = "AT";
	private static final String OPINION_GRA = "GRA";
	private static final String OPINION_MGA = "MGA";
	protected static final String OPINION_FIXED = "fixed";
	private static final String OPINION_LEVEL = "level";
	private static final String OPINION_SEA = "sea";

	/** The day of the month as a decimal number (range 01 to 31). */
	private static final String DAY_PAD_VAR = "%d";
	/** The day of the month as a decimal number (range 1 to 31). */
	private static final String DAY_VAR = "%-e";
	/** The full month name according to the current locale. */
	private static final String MONTH_VAR = "%B";
	/** The year as a decimal number including the century. */
	private static final String YEAR_VAR = "%Y";

	protected final LayoutInflater mInflater;
	protected final ZmanimSettings mSettings;
	protected final ComplexZmanimCalendar mCalendar;
	protected boolean mInIsrael;
	protected long mNow = System.currentTimeMillis();
	protected boolean mSummaries;
	protected boolean mElapsed;
	private DateFormat mTimeFormat;
	private Comparator<ZmanimItem> mComparator;
	private HebrewDateFormatter mHebrewDateFormatter;
	private String[] mMonthNames;
	private String mMonthDayYear;

	/**
	 * Time row item.
	 */
	protected static class ZmanimItem implements Comparable<ZmanimItem> {

		/** The row id. */
		public int rowId;
		/** The title id. */
		public int titleId;
		/** The summary. */
		public CharSequence summary;
		/** The time text id. */
		public int timeId;
		/** The time. */
		public Date time;
		/** The time label. */
		public CharSequence timeLabel;
		/** Has the time elapsed? */
		public boolean elapsed;

		/**
		 * Creates a new row item.
		 */
		public ZmanimItem() {
			super();
		}

		@Override
		public int compareTo(ZmanimItem that) {
			Date time1 = this.time;
			Date time2 = that.time;
			long t1 = (time1 == null) ? 0 : time1.getTime();
			long t2 = (time2 == null) ? 0 : time2.getTime();
			if (t1 != t2)
				return (t1 < t2) ? -1 : +1;
			int c = this.rowId - that.rowId;
			if (c != 0)
				return c;
			return this.titleId - that.titleId;
		}
	}

	/**
	 * Compare two time items.
	 */
	protected static class ZmanimComparator implements Comparator<ZmanimItem> {
		@Override
		public int compare(ZmanimItem lhs, ZmanimItem rhs) {
			return lhs.compareTo(rhs);
		}
	}

	/**
	 * Creates a new adapter.
	 * 
	 * @param context
	 *            the context.
	 * @param settings
	 *            the application settings.
	 */
	public ZmanimAdapter(Context context, ZmanimSettings settings) {
		super(context, R.layout.times_item, 0);
		mInflater = LayoutInflater.from(context);
		mSettings = settings;
		mCalendar = new ComplexZmanimCalendar();

		if (settings.isSeconds()) {
			boolean time24 = android.text.format.DateFormat.is24HourFormat(context);
			String pattern = context.getString(time24 ? R.string.twenty_four_hour_time_format : R.string.twelve_hour_time_format);
			mTimeFormat = new SimpleDateFormat(pattern, Locale.getDefault());
		} else {
			mTimeFormat = android.text.format.DateFormat.getTimeFormat(context);
		}
	}

	/**
	 * Get the calendar.
	 * 
	 * @return the calendar.
	 */
	public ComplexZmanimCalendar getCalendar() {
		return mCalendar;
	}

	@Override
	public View getView(int position, View convertView, ViewGroup parent) {
		return createViewFromResource(position, convertView, parent, R.layout.times_item);
	}

	/**
	 * Bind the item to the view.
	 * 
	 * @param position
	 *            the row index.
	 * @param convertView
	 *            the view.
	 * @param parent
	 *            the parent view.
	 * @param resource
	 *            the resource layout.
	 * @return the item view.
	 */
	protected View createViewFromResource(int position, View convertView, ViewGroup parent, int resource) {
		ZmanimItem item = getItem(position);
		boolean enabled = !item.elapsed;

		View view = convertView;
		ViewHolder holder;
		TextView title;
		TextView summary;
		TextView time;

		if (view == null) {
			view = mInflater.inflate(resource, parent, false);

			title = (TextView) view.findViewById(android.R.id.title);
			summary = (TextView) view.findViewById(android.R.id.summary);
			time = (TextView) view.findViewById(R.id.time);

			holder = new ViewHolder(title, summary, time);
			view.setTag(holder);
		} else {
			holder = (ViewHolder) view.getTag();
			title = holder.title;
			summary = holder.summary;
			time = holder.time;
		}
		view.setEnabled(enabled);
		view.setTag(R.id.time, item);

		title.setText(item.titleId);
		title.setEnabled(enabled);

		if (summary != null) {
			summary.setText(item.summary);
			summary.setEnabled(enabled);
			if (!mSummaries || (item.summary == null))
				summary.setVisibility(View.GONE);
		}

		time.setText(item.timeLabel);
		time.setEnabled(enabled);

		return view;
	}

	/**
	 * Adds the item to the array for a valid time.
	 * 
	 * @param titleId
	 *            the title label id.
	 * @param summaryId
	 *            the summary label id.
	 * @param time
	 *            the time.
	 */
	public void add(int titleId, int summaryId, Date time) {
		add(titleId, (summaryId == 0) ? (CharSequence) null : getContext().getText(summaryId), time);
	}

	/**
	 * Adds the item to the array for a valid time.
	 * 
	 * @param titleId
	 *            the title label id.
	 * @param summary
	 *            the summary label.
	 * @param time
	 *            the time in milliseconds.
	 */
	public void add(int titleId, CharSequence summary, Date time) {
		add(0, titleId, summary, titleId, time);
	}

	/**
	 * Adds the item to the array for a valid date.
	 * 
	 * @param rowId
	 *            the row id for remote views?
	 * @param titleId
	 *            the row layout id.
	 * @param timeId
	 *            the time text id to set for remote views.
	 * @param time
	 *            the time.
	 */
	public void add(int rowId, int titleId, int timeId, Date time) {
		add(rowId, titleId, null, timeId, time);
	}

	/**
	 * Adds the item to the array for a valid date.
	 * 
	 * @param rowId
	 *            the row id for remote views?
	 * @param titleId
	 *            the row layout id.
	 * @param summary
	 *            the summary label.
	 * @param timeId
	 *            the time text id to set for remote views.
	 * @param time
	 *            the time.
	 */
	private void add(int rowId, int titleId, CharSequence summary, int timeId, Date time) {
		ZmanimItem item = new ZmanimItem();
		item.rowId = rowId;
		item.titleId = titleId;
		item.summary = summary;
		item.timeId = timeId;
		item.time = time;

		if (time == null) {
			item.timeLabel = null;
			item.elapsed = true;
		} else {
			long t = time.getTime();
			item.timeLabel = mTimeFormat.format(time);
			if (rowId != 0)
				item.elapsed = (t < mNow);
			else
				item.elapsed = mElapsed ? false : (t < mNow);
		}

		if ((time != null) || (rowId != 0)) {
			add(item);
		}
	}

	protected void prePopulate() {
		clear();

		ZmanimSettings settings = mSettings;

		mSummaries = settings.isSummaries();
		mElapsed = settings.isPast();

		Context context = getContext();
		if (settings.isSeconds()) {
			boolean time24 = android.text.format.DateFormat.is24HourFormat(context);
			String pattern = context.getString(time24 ? R.string.twenty_four_hour_time_format : R.string.twelve_hour_time_format);
			mTimeFormat = new SimpleDateFormat(pattern, Locale.getDefault());
		} else {
			mTimeFormat = android.text.format.DateFormat.getTimeFormat(context);
		}

		mNow = System.currentTimeMillis();
	}

	/**
	 * Populate the list of times.
	 * 
	 * @param remote
	 *            is for remote views?
	 */
	public void populate(boolean remote) {
		prePopulate();

		ComplexZmanimCalendar cal = mCalendar;
		Calendar gcal = cal.getCalendar();
		JewishCalendar jcal = new JewishCalendar(gcal);
		jcal.setInIsrael(mInIsrael);
		int candlesOffset = mSettings.getCandleLightingOffset();
		int candles = getCandles(jcal);
		int candlesCount = candles & CANDLES_MASK;
		boolean hasCandles = candlesCount > 0;
		int candlesHow = candles & MOTZE_MASK;
		int holidayTomorrow = (candles >> 4) & HOLIDAY_MASK;
		int holidayToday = (candles >> 12) & HOLIDAY_MASK;
		Date dateCandles;

		Date date;
		int summary;
		String opinion;
		final Resources res = getContext().getResources();

		opinion = mSettings.getDawn();
		if (OPINION_19_8.equals(opinion)) {
			date = cal.getAlos19Point8Degrees();
			summary = R.string.dawn_19;
		} else if (OPINION_120.equals(opinion)) {
			date = cal.getAlos120();
			summary = R.string.dawn_120;
		} else if (OPINION_120_ZMANIS.equals(opinion)) {
			date = cal.getAlos120Zmanis();
			summary = R.string.dawn_120_zmanis;
		} else if (OPINION_18.equals(opinion)) {
			date = cal.getAlos18Degrees();
			summary = R.string.dawn_18;
		} else if (OPINION_26.equals(opinion)) {
			date = cal.getAlos26Degrees();
			summary = R.string.dawn_26;
		} else if (OPINION_16_1.equals(opinion)) {
			date = cal.getAlos16Point1Degrees();
			summary = R.string.dawn_16;
		} else if (OPINION_96.equals(opinion)) {
			date = cal.getAlos96();
			summary = R.string.dawn_96;
		} else if (OPINION_96_ZMANIS.equals(opinion)) {
			date = cal.getAlos90Zmanis();
			summary = R.string.dawn_96_zmanis;
		} else if (OPINION_90.equals(opinion)) {
			date = cal.getAlos90();
			summary = R.string.dawn_90;
		} else if (OPINION_90_ZMANIS.equals(opinion)) {
			date = cal.getAlos90Zmanis();
			summary = R.string.dawn_90_zmanis;
		} else if (OPINION_72.equals(opinion)) {
			date = cal.getAlos72();
			summary = R.string.dawn_72;
		} else if (OPINION_72_ZMANIS.equals(opinion)) {
			date = cal.getAlos72Zmanis();
			summary = R.string.dawn_72_zmanis;
		} else if (OPINION_60.equals(opinion)) {
			date = cal.getAlos60();
			summary = R.string.dawn_60;
		} else {
			date = cal.getAlosHashachar();
			summary = R.string.dawn_16;
		}
		if (date == null) {
			date = cal.getAlos120Zmanis();
			summary = R.string.dawn_120_zmanis;
		}
		if (remote)
			add(R.id.dawn_row, R.string.dawn, R.id.dawn_time, date);
		else
			add(R.string.dawn, summary, date);
		if ((holidayToday == JewishCalendar.SEVENTEEN_OF_TAMMUZ) || (holidayToday == JewishCalendar.FAST_OF_GEDALYAH) || (holidayToday == JewishCalendar.TENTH_OF_TEVES)
				|| (holidayToday == JewishCalendar.FAST_OF_ESTHER)) {
			add(R.string.fast_begins, null, date);
		}

		opinion = mSettings.getTallis();
		if (OPINION_10_2.equals(opinion)) {
			date = cal.getMisheyakir10Point2Degrees();
			summary = R.string.tallis_10;
		} else if (OPINION_11.equals(opinion)) {
			date = cal.getMisheyakir11Degrees();
			summary = R.string.tallis_11;
		} else {
			date = cal.getMisheyakir11Point5Degrees();
			summary = R.string.tallis_summary;
		}
		if (remote)
			add(R.id.tallis_row, R.string.tallis, R.id.tallis_time, date);
		else
			add(R.string.tallis, summary, date);

		opinion = mSettings.getSunrise();
		if (OPINION_SEA.equals(opinion)) {
			date = cal.getSeaLevelSunrise();
			summary = R.string.sunrise_sea;
		} else {
			date = cal.getSunrise();
			summary = R.string.sunrise_summary;
		}
		if (remote)
			add(R.id.sunrise_row, R.string.sunrise, R.id.sunrise_time, date);
		else
			add(R.string.sunrise, summary, date);

		opinion = mSettings.getLastShema();
		if (OPINION_16_1_SUNSET.equals(opinion)) {
			date = cal.getSofZmanShmaAlos16Point1ToSunset();
			summary = R.string.shema_16_sunset;
		} else if (OPINION_7_083.equals(opinion)) {
			date = cal.getSofZmanShmaAlos16Point1ToTzaisGeonim7Point083Degrees();
			summary = R.string.shema_7;
		} else if (OPINION_19_8.equals(opinion)) {
			date = cal.getSofZmanShmaMGA19Point8Degrees();
			summary = R.string.shema_19;
		} else if (OPINION_120.equals(opinion)) {
			date = cal.getSofZmanShmaMGA120Minutes();
			summary = R.string.shema_120;
		} else if (OPINION_18.equals(opinion)) {
			date = cal.getSofZmanShmaMGA18Degrees();
			summary = R.string.shema_18;
		} else if (OPINION_96.equals(opinion)) {
			date = cal.getSofZmanShmaMGA96Minutes();
			summary = R.string.shema_96;
		} else if (OPINION_96_ZMANIS.equals(opinion)) {
			date = cal.getSofZmanShmaMGA96MinutesZmanis();
			summary = R.string.shema_96_zmanis;
		} else if (OPINION_16_1.equals(opinion)) {
			date = cal.getSofZmanShmaMGA16Point1Degrees();
			summary = R.string.shema_16;
		} else if (OPINION_90.equals(opinion)) {
			date = cal.getSofZmanShmaMGA90Minutes();
			summary = R.string.shema_90;
		} else if (OPINION_90_ZMANIS.equals(opinion)) {
			date = cal.getSofZmanShmaMGA90MinutesZmanis();
			summary = R.string.shema_90_zmanis;
		} else if (OPINION_72.equals(opinion)) {
			date = cal.getSofZmanShmaMGA72Minutes();
			summary = R.string.shema_72;
		} else if (OPINION_72_ZMANIS.equals(opinion)) {
			date = cal.getSofZmanShmaMGA72MinutesZmanis();
			summary = R.string.shema_72_zmanis;
		} else if (OPINION_MGA.equals(opinion)) {
			date = cal.getSofZmanShmaMGA();
			summary = R.string.shema_mga;
		} else if (OPINION_ATERET.equals(opinion)) {
			date = cal.getSofZmanShmaAteretTorah();
			summary = R.string.shema_ateret;
		} else if (OPINION_3.equals(opinion)) {
			date = cal.getSofZmanShma3HoursBeforeChatzos();
			summary = R.string.shema_3;
		} else if (OPINION_FIXED.equals(opinion)) {
			date = cal.getSofZmanShmaFixedLocal();
			summary = R.string.shema_fixed;
		} else if (OPINION_GRA.equals(opinion)) {
			date = cal.getSofZmanShmaGRA();
			summary = R.string.shema_gra;
		} else {
			date = cal.getSofZmanShmaMGA();
			summary = R.string.shema_mga;
		}
		if (remote)
			add(R.id.shema_row, R.string.shema, R.id.shema_time, date);
		else
			add(R.string.shema, summary, date);

		opinion = mSettings.getLastTfila();
		if (OPINION_120.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA120Minutes();
			summary = R.string.prayers_120;
		} else if (OPINION_96.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA96Minutes();
			summary = R.string.prayers_96;
		} else if (OPINION_96_ZMANIS.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA96MinutesZmanis();
			summary = R.string.prayers_96_zmanis;
		} else if (OPINION_19_8.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA19Point8Degrees();
			summary = R.string.prayers_19;
		} else if (OPINION_90.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA90Minutes();
			summary = R.string.prayers_90;
		} else if (OPINION_90_ZMANIS.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA90MinutesZmanis();
			summary = R.string.prayers_90_zmanis;
		} else if (OPINION_ATERET.equals(opinion)) {
			date = cal.getSofZmanTfilahAteretTorah();
			summary = R.string.prayers_ateret;
		} else if (OPINION_18.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA18Degrees();
			summary = R.string.prayers_18;
		} else if (OPINION_FIXED.equals(opinion)) {
			date = cal.getSofZmanTfilaFixedLocal();
			summary = R.string.prayers_fixed;
		} else if (OPINION_16_1.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA16Point1Degrees();
			summary = R.string.prayers_16;
		} else if (OPINION_72.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA72Minutes();
			summary = R.string.prayers_72;
		} else if (OPINION_72_ZMANIS.equals(opinion)) {
			date = cal.getSofZmanTfilaMGA72MinutesZmanis();
			summary = R.string.prayers_72_zmanis;
		} else if (OPINION_2.equals(opinion)) {
			date = cal.getSofZmanTfila2HoursBeforeChatzos();
			summary = R.string.prayers_2;
		} else if (OPINION_GRA.equals(opinion)) {
			date = cal.getSofZmanTfilaGRA();
			summary = R.string.prayers_gra;
		} else {
			date = cal.getSofZmanTfilaMGA();
			summary = R.string.prayers_mga;
		}
		if (remote)
			add(R.id.prayers_row, R.string.prayers, R.id.prayers_time, date);
		else {
			add(R.string.prayers, summary, date);
			if (holidayToday == JewishCalendar.EREV_PESACH)
				add(R.string.eat_chametz, summary, date);
		}

		if (!remote && (holidayToday == JewishCalendar.EREV_PESACH)) {
			opinion = mSettings.getBurnChametz();
			if (OPINION_16_1.equals(opinion)) {
				date = cal.getSofZmanBiurChametzMGA16Point1Degrees();
				summary = R.string.burn_chametz_16;
			} else if (OPINION_72.equals(opinion)) {
				date = cal.getSofZmanBiurChametzMGA72Minutes();
				summary = R.string.burn_chametz_72;
			} else {
				date = cal.getSofZmanBiurChametzGRA();
				summary = R.string.burn_chametz_gra;
			}
			add(R.string.burn_chametz, summary, date);
		}

		opinion = mSettings.getMidday();
		if (OPINION_FIXED.equals(opinion)) {
			date = cal.getFixedLocalChatzos();
			summary = R.string.midday_fixed;
		} else {
			date = cal.getChatzos();
			summary = R.string.midday_summary;
		}
		if (remote)
			add(R.id.midday_row, R.string.midday, R.id.midday_time, date);
		else
			add(R.string.midday, summary, date);
		Date midday = date;

		opinion = mSettings.getEarliestMincha();
		if (OPINION_16_1.equals(opinion)) {
			date = cal.getMinchaGedola16Point1Degrees();
			summary = R.string.earliest_mincha_16;
		} else if (OPINION_30.equals(opinion)) {
			date = cal.getMinchaGedola30Minutes();
			summary = R.string.earliest_mincha_30;
		} else if (OPINION_ATERET.equals(opinion)) {
			date = cal.getMinchaGedolaAteretTorah();
			summary = R.string.earliest_mincha_ateret;
		} else if (OPINION_72.equals(opinion)) {
			date = cal.getMinchaGedola72Minutes();
			summary = R.string.earliest_mincha_72;
		} else {
			date = cal.getMinchaGedola();
			summary = R.string.earliest_mincha_summary;
		}
		if (remote)
			add(R.id.earliest_mincha_row, R.string.earliest_mincha, R.id.earliest_mincha_time, date);
		else
			add(R.string.earliest_mincha, summary, date);

		opinion = mSettings.getMincha();
		if (OPINION_16_1.equals(opinion)) {
			date = cal.getMinchaKetana16Point1Degrees();
			summary = R.string.mincha_16;
		} else if (OPINION_72.equals(opinion)) {
			date = cal.getMinchaKetana72Minutes();
			summary = R.string.mincha_72;
		} else if (OPINION_ATERET.equals(opinion)) {
			date = cal.getMinchaKetanaAteretTorah();
			summary = R.string.mincha_ateret;
		} else {
			date = cal.getMinchaKetana();
			summary = R.string.mincha_summary;
		}
		if (remote)
			add(R.id.mincha_row, R.string.mincha, R.id.mincha_time, date);
		else
			add(R.string.mincha, summary, date);

		opinion = mSettings.getPlugHamincha();
		if (OPINION_16_1_SUNSET.equals(opinion)) {
			date = cal.getPlagAlosToSunset();
			summary = R.string.plug_hamincha_16_sunset;
		} else if (OPINION_16_1_ALOS.equals(opinion)) {
			date = cal.getPlagAlos16Point1ToTzaisGeonim7Point083Degrees();
			summary = R.string.plug_hamincha_16_alos;
		} else if (OPINION_ATERET.equals(opinion)) {
			date = cal.getPlagHaminchaAteretTorah();
			summary = R.string.plug_hamincha_ateret;
		} else if (OPINION_60.equals(opinion)) {
			date = cal.getPlagHamincha60Minutes();
			summary = R.string.plug_hamincha_60;
		} else if (OPINION_72.equals(opinion)) {
			date = cal.getPlagHamincha72Minutes();
			summary = R.string.plug_hamincha_72;
		} else if (OPINION_72_ZMANIS.equals(opinion)) {
			date = cal.getPlagHamincha72MinutesZmanis();
			summary = R.string.plug_hamincha_72_zmanis;
		} else if (OPINION_16_1.equals(opinion)) {
			date = cal.getPlagHamincha16Point1Degrees();
			summary = R.string.plug_hamincha_16;
		} else if (OPINION_18.equals(opinion)) {
			date = cal.getPlagHamincha18Degrees();
			summary = R.string.plug_hamincha_18;
		} else if (OPINION_90.equals(opinion)) {
			date = cal.getPlagHamincha90Minutes();
			summary = R.string.plug_hamincha_90;
		} else if (OPINION_90_ZMANIS.equals(opinion)) {
			date = cal.getPlagHamincha90MinutesZmanis();
			summary = R.string.plug_hamincha_90_zmanis;
		} else if (OPINION_19_8.equals(opinion)) {
			date = cal.getPlagHamincha19Point8Degrees();
			summary = R.string.plug_hamincha_19;
		} else if (OPINION_96.equals(opinion)) {
			date = cal.getPlagHamincha96Minutes();
			summary = R.string.plug_hamincha_96;
		} else if (OPINION_96_ZMANIS.equals(opinion)) {
			date = cal.getPlagHamincha96MinutesZmanis();
			summary = R.string.plug_hamincha_96_zmanis;
		} else if (OPINION_120.equals(opinion)) {
			date = cal.getPlagHamincha120Minutes();
			summary = R.string.plug_hamincha_120;
		} else if (OPINION_120_ZMANIS.equals(opinion)) {
			date = cal.getPlagHamincha120MinutesZmanis();
			summary = R.string.plug_hamincha_120_zmanis;
		} else if (OPINION_26.equals(opinion)) {
			date = cal.getPlagHamincha26Degrees();
			summary = R.string.plug_hamincha_26;
		} else {
			date = cal.getPlagHamincha();
			summary = R.string.plug_hamincha_gra;
		}
		if (remote)
			add(R.id.plug_hamincha_row, R.string.plug_hamincha, R.id.plug_hamincha_time, date);
		else
			add(R.string.plug_hamincha, summary, date);

		opinion = mSettings.getSunset();
		if (OPINION_LEVEL.equals(opinion)) {
			date = cal.getSunset();
			summary = R.string.sunset_summary;
		} else {
			date = cal.getSeaLevelSunset();
			summary = R.string.sunset_sea;
		}
		if (hasCandles && (candlesHow == BEFORE_SUNSET)) {
			dateCandles = cal.getTimeOffset(date, -candlesOffset * DateUtils.MINUTE_IN_MILLIS);
			if (remote) {
				add(R.id.candles_row, R.string.candles, R.id.candles_time, dateCandles);
			} else {
				String summaryText;
				if (holidayTomorrow == JewishCalendar.CHANUKAH) {
					summaryText = res.getQuantityString(R.plurals.candles_chanukka, candlesCount, candlesCount);
				} else {
					summaryText = res.getQuantityString(R.plurals.candles_summary, candlesOffset, candlesOffset);
				}
				add(R.string.candles, summaryText, dateCandles);
			}
		} else if (remote) {
			add(R.id.candles_row, R.string.candles, R.id.candles_time, null);
		}

		if (hasCandles && (candlesHow == AT_SUNSET)) {
			if (remote) {
				add(R.id.candles_row, R.string.candles, R.id.candles_time, date);
			} else if (holidayTomorrow == JewishCalendar.CHANUKAH) {
				String summaryText = res.getQuantityString(R.plurals.candles_chanukka, candlesCount, candlesCount);
				add(R.string.candles, summaryText, date);
			} else {
				add(R.string.candles, summary, date);
			}
		} else if (remote) {
			add(R.id.candles_row, R.string.candles, R.id.candles_time, null);
		}
		if ((holidayTomorrow == JewishCalendar.TISHA_BEAV) || (holidayTomorrow == JewishCalendar.YOM_KIPPUR)) {
			add(R.string.fast_begins, null, date);
		}
		if (remote)
			add(R.id.sunset_row, R.string.sunset, R.id.sunset_time, date);
		else
			add(R.string.sunset, summary, date);

		opinion = mSettings.getTwilight();
		if (OPINION_7_083.equals(opinion)) {
			date = cal.getBainHasmashosRT13Point5MinutesBefore7Point083Degrees();
			summary = R.string.twilight_7_083;
		} else if (OPINION_58.equals(opinion)) {
			date = cal.getBainHasmashosRT58Point5Minutes();
			summary = R.string.twilight_58;
		} else if (OPINION_13.equals(opinion)) {
			date = cal.getBainHasmashosRT13Point24Degrees();
			summary = R.string.twilight_13;
		} else {
			date = cal.getBainHasmashosRT2Stars();
			summary = R.string.twilight_2stars;
			if (date == null) {
				date = cal.getBainHasmashosRT13Point5MinutesBefore7Point083Degrees();
				summary = R.string.twilight_7_083;
			}
		}
		if (remote)
			add(R.id.twilight_row, R.string.twilight, R.id.twilight_time, date);
		else
			add(R.string.twilight, summary, date);
		if ((holidayToday == JewishCalendar.SEVENTEEN_OF_TAMMUZ) || (holidayToday == JewishCalendar.TISHA_BEAV) || (holidayToday == JewishCalendar.FAST_OF_GEDALYAH)
				|| (holidayToday == JewishCalendar.TENTH_OF_TEVES) || (holidayToday == JewishCalendar.FAST_OF_ESTHER)) {
			add(R.string.fast_ends, null, date);
		}

		opinion = mSettings.getNightfall();
		if (OPINION_120.equals(opinion)) {
			date = cal.getTzais120();
			summary = R.string.nightfall_120;
		} else if (OPINION_120_ZMANIS.equals(opinion)) {
			date = cal.getTzais120Zmanis();
			summary = R.string.nightfall_120_zmanis;
		} else if (OPINION_16_1.equals(opinion)) {
			date = cal.getTzais16Point1Degrees();
			summary = R.string.nightfall_16;
		} else if (OPINION_18.equals(opinion)) {
			date = cal.getTzais18Degrees();
			summary = R.string.nightfall_18;
		} else if (OPINION_19_8.equals(opinion)) {
			date = cal.getTzais19Point8Degrees();
			summary = R.string.nightfall_19;
		} else if (OPINION_26.equals(opinion)) {
			date = cal.getTzais26Degrees();
			summary = R.string.nightfall_26;
		} else if (OPINION_60.equals(opinion)) {
			date = cal.getTzais60();
			summary = R.string.nightfall_60;
		} else if (OPINION_72.equals(opinion)) {
			date = cal.getTzais72();
			summary = R.string.nightfall_72;
		} else if (OPINION_72_ZMANIS.equals(opinion)) {
			date = cal.getTzais72Zmanis();
			summary = R.string.nightfall_72_zmanis;
		} else if (OPINION_90.equals(opinion)) {
			date = cal.getTzais90();
			summary = R.string.nightfall_90;
		} else if (OPINION_90_ZMANIS.equals(opinion)) {
			date = cal.getTzais90Zmanis();
			summary = R.string.nightfall_90_zmanis;
		} else if (OPINION_96.equals(opinion)) {
			date = cal.getTzais96();
			summary = R.string.nightfall_96;
		} else if (OPINION_96_ZMANIS.equals(opinion)) {
			date = cal.getTzais96Zmanis();
			summary = R.string.nightfall_96_zmanis;
		} else if (OPINION_ATERET.equals(opinion)) {
			date = cal.getTzaisAteretTorah();
			summary = R.string.nightfall_ateret;
		} else if (OPINION_3_65.equals(opinion)) {
			date = cal.getTzaisGeonim3Point65Degrees();
			summary = R.string.nightfall_3_65;
		} else if (OPINION_3_676.equals(opinion)) {
			date = cal.getTzaisGeonim3Point676Degrees();
			summary = R.string.nightfall_3_676;
		} else if (OPINION_4_37.equals(opinion)) {
			date = cal.getTzaisGeonim4Point37Degrees();
			summary = R.string.nightfall_4_37;
		} else if (OPINION_4_61.equals(opinion)) {
			date = cal.getTzaisGeonim4Point61Degrees();
			summary = R.string.nightfall_4_61;
		} else if (OPINION_4_8.equals(opinion)) {
			date = cal.getTzaisGeonim4Point8Degrees();
			summary = R.string.nightfall_4_8;
		} else if (OPINION_5_88.equals(opinion)) {
			date = cal.getTzaisGeonim5Point88Degrees();
			summary = R.string.nightfall_5_88;
		} else if (OPINION_5_95.equals(opinion)) {
			date = cal.getTzaisGeonim5Point95Degrees();
			summary = R.string.nightfall_5_95;
		} else if (OPINION_7_083.equals(opinion)) {
			date = cal.getTzaisGeonim7Point083Degrees();
			summary = R.string.nightfall_7;
		} else if (OPINION_8_5.equals(opinion)) {
			date = cal.getTzaisGeonim8Point5Degrees();
			summary = R.string.nightfall_8;
		} else {
			date = cal.getTzais();
			summary = R.string.nightfall_3stars;
		}
		if (remote)
			add(R.id.nightfall_row, R.string.nightfall, R.id.nightfall_time, date);
		else
			add(R.string.nightfall, summary, date);
		if (holidayToday == JewishCalendar.YOM_KIPPUR) {
			add(R.string.fast_ends, null, date);
		}
		if (hasCandles && (candlesHow == MOTZE_SHABBATH)) {
			if (remote) {
				add(R.id.candles_nightfall_row, R.string.nightfall, R.id.candles_nightfall_time, date);
			} else if (holidayTomorrow == JewishCalendar.CHANUKAH) {
				String summaryText = res.getQuantityString(R.plurals.candles_chanukka, candlesCount, candlesCount);
				add(R.string.candles, summaryText, date);
			} else {
				add(R.string.candles, summary, date);
			}
		} else if (remote) {
			add(R.id.candles_nightfall_row, R.string.nightfall, R.id.candles_nightfall_time, null);
		}

		opinion = mSettings.getMidnight();
		if (OPINION_12.equals(opinion)) {
			date = midday;
			if (date != null)
				date.setTime(date.getTime() + TWELVE_HOURS);
			summary = R.string.midnight_12;
		} else {
			date = cal.getSolarMidnight();
			summary = R.string.midnight_summary;
		}
		if (remote)
			add(R.id.midnight_row, R.string.midnight, R.id.midnight_time, date);
		else
			add(R.string.midnight, summary, date);

		if (!remote) {
			final int jDayOfMonth = jcal.getJewishDayOfMonth();
			// Molad.
			if ((jDayOfMonth <= 2) || (jDayOfMonth >= 25)) {
				int y = gcal.get(Calendar.YEAR);
				int m = gcal.get(Calendar.MONTH);
				int d = gcal.get(Calendar.DAY_OF_MONTH);
				jcal.forward();// Molad is always of the previous month.
				JewishDate molad = jcal.getMolad();
				int moladYear = molad.getGregorianYear();
				int moladMonth = molad.getGregorianMonth();
				int moladDay = molad.getGregorianDayOfMonth();
				if ((moladYear == y) && (moladMonth == m) && (moladDay == d)) {
					double moladSeconds = (molad.getMoladChalakim() * 10.0) / 3.0;
					double moladSecondsFloor = Math.floor(moladSeconds);
					Calendar calMolad = (Calendar) gcal.clone();
					calMolad.set(moladYear, moladMonth, moladDay, molad.getMoladHours(), molad.getMoladMinutes(), (int) moladSecondsFloor);
					calMolad.set(Calendar.MILLISECOND, (int) (DateUtils.SECOND_IN_MILLIS * (moladSeconds - moladSecondsFloor)));
					summary = R.string.molad_summary;
					add(R.string.molad, summary, calMolad.getTime());
				}
			}
			// First Kiddush Levana.
			else if ((jDayOfMonth >= 2) && (jDayOfMonth <= 8)) {
				opinion = mSettings.getEarliestKiddushLevana();
				if (OPINION_7.equals(opinion)) {
					date = cal.getTchilasZmanKidushLevana7Days();
					summary = R.string.levana_7;
				} else {
					date = cal.getTchilasZmanKidushLevana3Days();
					summary = R.string.levana_earliest_summary;
				}
				add(R.string.levana_earliest, summary, date);
			}
			// Last Kiddush Levana.
			else if ((jDayOfMonth > 10) && (jDayOfMonth < 20)) {
				opinion = mSettings.getLatestKiddushLevana();
				if (OPINION_15.equals(opinion)) {
					date = cal.getSofZmanKidushLevana15Days();
					summary = R.string.levana_15;
				} else {
					date = cal.getSofZmanKidushLevanaBetweenMoldos();
					summary = R.string.levana_latest_summary;
				}
				add(R.string.levana_latest, summary, date);
			}
		}

		sort();
	}

	/**
	 * Get the number of candles to light.
	 * 
	 * @param cal
	 *            the Gregorian date.
	 * @param inIsrael
	 *            is in Israel?
	 * @return the number of candles to light, the holiday, and when to light.
	 */
	protected int getCandles(JewishCalendar jcal) {
		final int dayOfWeek = jcal.getDayOfWeek();

		// Check if the following day is special, because we can't check
		// EREV_CHANUKAH.
		int holidayToday = jcal.getYomTovIndex();
		jcal.forward();
		int holidayTomorrow = jcal.getYomTovIndex();
		int count = CANDLES_NONE;
		int flags = BEFORE_SUNSET;

		switch (holidayTomorrow) {
		case JewishCalendar.ROSH_HASHANA:
		case JewishCalendar.SUCCOS:
		case JewishCalendar.SHEMINI_ATZERES:
		case JewishCalendar.SIMCHAS_TORAH:
		case JewishCalendar.PESACH:
		case JewishCalendar.SHAVUOS:
			count = CANDLES_FESTIVAL;
			break;
		case JewishCalendar.YOM_KIPPUR:
			count = CANDLES_YOM_KIPPUR;
			break;
		case JewishCalendar.CHANUKAH:
			count = jcal.getDayOfChanukah();
			if ((dayOfWeek != Calendar.FRIDAY) && (dayOfWeek != Calendar.SATURDAY))
				flags = AT_SUNSET;
			break;
		default:
			if (dayOfWeek == Calendar.FRIDAY) {
				holidayTomorrow = SHABBATH;
				count = CANDLES_SHABBATH;
			}
			break;
		}

		// Forbidden to light candles during Shabbath.
		switch (dayOfWeek) {
		case Calendar.FRIDAY:
			// Probably never happens that Yom Kippurim falls on a Friday.
			// Prohibited to light candles on Yom Kippurim for Shabbath.
			if (holidayToday == JewishCalendar.YOM_KIPPUR) {
				count = CANDLES_NONE;
			}
			break;
		case Calendar.SATURDAY:
			if (holidayToday == -1) {
				holidayToday = SHABBATH;
			}
			flags = MOTZE_SHABBATH;
			break;
		default:
			// During a holiday, we can light for the next day from an existing
			// flame.
			switch (holidayToday) {
			case JewishCalendar.ROSH_HASHANA:
			case JewishCalendar.SUCCOS:
			case JewishCalendar.SHEMINI_ATZERES:
			case JewishCalendar.SIMCHAS_TORAH:
			case JewishCalendar.PESACH:
			case JewishCalendar.SHAVUOS:
				flags = AT_SUNSET;
				break;
			}
			break;
		}

		return flags | ((holidayToday & HOLIDAY_MASK) << 12) | ((holidayTomorrow & HOLIDAY_MASK) << 4) | (count & CANDLES_MASK);
	}

	/**
	 * Sort.
	 */
	protected void sort() {
		if (mComparator == null) {
			mComparator = new ZmanimComparator();
		}
		sort(mComparator);
	}

	/**
	 * View holder for zman row item.
	 * 
	 * @author Moshe W
	 */
	private static class ViewHolder {

		public final TextView title;
		public final TextView summary;
		public final TextView time;

		public ViewHolder(TextView title, TextView summary, TextView time) {
			this.title = title;
			this.summary = summary;
			this.time = time;
		}
	}

	public CharSequence formatDate(JewishDate jewishDate) {
		int jewishDay = jewishDate.getJewishDayOfMonth();
		int jewishMonth = jewishDate.getJewishMonth();
		int jewishYear = jewishDate.getJewishYear();
		if ((jewishMonth == JewishDate.ADAR) && jewishDate.isJewishLeapYear()) {
			jewishMonth = 14; // return Adar I, not Adar in a leap year
		}

		String[] monthNames = mMonthNames;
		if (monthNames == null) {
			monthNames = getContext().getResources().getStringArray(R.array.hebrew_months);
			mMonthNames = monthNames;
		}
		String format = mMonthDayYear;
		if (format == null) {
			format = getContext().getString(R.string.month_day_year);
			mMonthDayYear = format;
		}

		String yearStr = null;
		String monthStr = monthNames[jewishMonth - 1];
		String dayStr = null;
		String dayPadded = null;

		if (ZmanimLocations.isLocaleRTL()) {
			HebrewDateFormatter formatter = mHebrewDateFormatter;
			if (formatter == null) {
				formatter = new HebrewDateFormatter();
				formatter.setHebrewFormat(true);
				mHebrewDateFormatter = formatter;
			}

			yearStr = formatter.formatHebrewNumber(jewishYear);
			dayStr = formatter.formatHebrewNumber(jewishDay);
			dayPadded = dayStr;
		} else {
			yearStr = String.valueOf(jewishYear);
			dayStr = String.valueOf(jewishDay);
			dayPadded = (jewishDay < 10) ? "0" + dayStr : dayStr;
		}

		String formatted = format.replaceAll(YEAR_VAR, yearStr);
		formatted = formatted.replaceAll(MONTH_VAR, monthStr);
		formatted = formatted.replaceAll(DAY_VAR, dayStr);
		formatted = formatted.replaceAll(DAY_PAD_VAR, dayPadded);

		return formatted;
	}

	/**
	 * Set the calendar.
	 * 
	 * @param calendar
	 *            the calendar.
	 */
	public void setCalendar(Calendar calendar) {
		this.mCalendar.setCalendar(calendar);
	}

	/**
	 * Set the calendar time.
	 * 
	 * @param time
	 *            the time in milliseconds.
	 */
	public void setCalendar(long time) {
		Calendar cal = mCalendar.getCalendar();
		cal.setTimeInMillis(time);
	}

	/**
	 * Sets the {@link GeoLocation}.
	 * 
	 * @param geoLocation
	 *            the location.
	 */
	public void setGeoLocation(GeoLocation geoLocation) {
		this.mCalendar.setGeoLocation(geoLocation);
	}

	/**
	 * Sets whether to use Israel holiday scheme or not.
	 * 
	 * @param inIsrael
	 *            set to {@code true} for calculations for Israel.
	 */
	public void setInIsrael(boolean inIsrael) {
		this.mInIsrael = inIsrael;
	}
}
